import { clsx, type ClassValue } from "clsx";
import { twMerge } from "tailwind-merge";
import { GameState } from "@/hooks/useGameState"; // Assuming GameState is exported from your hook

export function cn(...inputs: ClassValue[]) {
  return twMerge(clsx(inputs));
}

const PLAYER_MAIN_OUTCOME = "PLAYER";
const BANKER_MAIN_OUTCOME = "BANKER";

export const calculateUserScoreAndStats = (gameState: GameState) => {
  let correct = 0;
  let total = 0;

  Object.entries(gameState.userVotes).forEach(([round, vote]) => {
    const roundIndex = Number.parseInt(round) - 1;
    const result = gameState.results[roundIndex];

    if (result) {
      if (result === "TIE") {
        return;
      }
      total++;
      if (vote === "BANKER") {
        if (result === "BANKER" || result === "DRAGON") {
          correct++;
        }
      } else if (vote === "PLAYER") {
        if (result === "PLAYER" || result === "PANDA") {
          correct++;
        }
      }
    }
  });

  if (
    Object.keys(gameState.userVotes).length === 0 &&
    gameState.results.length === 0
  ) {
    correct = 0;
    total = 0;
  }

  const incorrect = total - correct;
  const sessionScore = correct - incorrect;

  const playerCount =
    gameState.results.filter((r) => r === PLAYER_MAIN_OUTCOME).length || 0;
  const bankerCount =
    gameState.results.filter((r) => r === BANKER_MAIN_OUTCOME).length || 0;
  const tieCount = gameState.results.filter((r) => r === "TIE").length || 0;
  const dragonCount =
    gameState.results.filter((r) => r === "DRAGON").length || 0;
  const pandaCount = gameState.results.filter((r) => r === "PANDA").length || 0;

  const stats = [
    {
      label: "PLAYER",
      count: playerCount,
      iconImage: "/images/player.png",
      alt: "Player",
    },
    {
      label: "BANKER",
      count: bankerCount,
      iconImage: "/images/banker.png",
      alt: "Banker",
    },
    {
      label: "DRAGONS",
      count: dragonCount,
      iconImage: "/images/dragonGIF.gif",
      alt: "Dragon",
      color: "#feda01",
    },
    {
      label: "PANDA",
      count: pandaCount,
      iconImage: "/images/pandaGIF.gif",
      alt: "Panda",
      color: "#3abefe",
    },
    {
      label: "TIES",
      count: tieCount,
      iconImage: "/images/TIEGif2.gif",
      alt: "Tie",
      color: "#58fe00",
    },
  ];

  return { correct, incorrect, sessionScore, stats };
};

export const calculateDisplayPercentages = (gameState: GameState) => {
  const bankerWins = gameState.results.filter(
    (r) => r === "BANKER" || r === "DRAGON"
  ).length;
  const playerWins = gameState.results.filter(
    (r) => r === "PLAYER" || r === "PANDA"
  ).length;
  const tieCount = gameState.results.filter((r) => r === "TIE").length || 0;
  const totalResults = bankerWins + playerWins + tieCount;

  const calculatedTiePercentage =
    totalResults > 0 ? Math.round((tieCount / totalResults) * 100) : 0;
  const displayTiePercentage = calculatedTiePercentage;

  const remainingPercentageForBankerPlayer = 100 - displayTiePercentage;
  const totalBankerAndPlayerWins = bankerWins + playerWins;

  let dynamicDisplayBankerPercentage;
  let dynamicDisplayPlayerPercentage;

  if (totalBankerAndPlayerWins > 0) {
    dynamicDisplayBankerPercentage = Math.round(
      (bankerWins / totalBankerAndPlayerWins) *
        remainingPercentageForBankerPlayer
    );
    dynamicDisplayPlayerPercentage =
      remainingPercentageForBankerPlayer - dynamicDisplayBankerPercentage;
  } else {
    dynamicDisplayBankerPercentage = Math.floor(
      remainingPercentageForBankerPlayer / 2
    );
    dynamicDisplayPlayerPercentage = Math.ceil(
      remainingPercentageForBankerPlayer / 2
    );
  }
  return {
    displayPlayerPercentage: dynamicDisplayPlayerPercentage,
    displayBankerPercentage: dynamicDisplayBankerPercentage,
    displayTiePercentage,
  };
};

export const calculateCommunityPercentages = (gameState: GameState) => {
  const totalCommunityVotes = gameState.votes.banker + gameState.votes.player;
  const communityBankerPercentage =
    totalCommunityVotes > 0
      ? Math.round((gameState.votes.banker / totalCommunityVotes) * 100)
      : 50;
  const communityPlayerPercentage =
    totalCommunityVotes > 0
      ? Math.round((gameState.votes.player / totalCommunityVotes) * 100)
      : 50;
  return { communityBankerPercentage, communityPlayerPercentage };
};
