"use client";

import { useState, useCallback, useEffect } from "react";

const EXPRESS_BACKEND_URL = "https://baccaratpt.com";

export interface GameState {
  results: string[];
  currentRound: number;
  maxRounds: number;
  votes: { player: number; banker: number };
  roundStartTime: number | null;
  userVoted: boolean;
  userVotes: Record<number, string>;
  bettingDurationSeconds: number | null;
}

export function useGameState() {
  const [gameState, setGameState] = useState<GameState>({
    results: [],
    currentRound: 1,
    maxRounds: 80,
    votes: { player: 0, banker: 0 },
    roundStartTime: null,
    userVoted: false,
    userVotes: {},
    bettingDurationSeconds: null,
  });

  const fetchGameState = useCallback(async () => {
    try {
      const response = await fetch(`${EXPRESS_BACKEND_URL}/api/game`, {
        cache: "no-store",
        headers: { "Cache-Control": "no-cache" },
      });

      if (!response.ok)
        throw new Error(`API responded with status: ${response.status}`);

      const data = await response.json();

      setGameState((prev) => {
        const newState = {
          ...prev,
          results: data.results,
          currentRound: data.currentRound,
          votes: data.votes,
          roundStartTime: data.roundStartTime,
          bettingDurationSeconds: data.bettingDurationSeconds,
        };

        const userVotesFromStorage = JSON.parse(
          sessionStorage.getItem("userVotes") || "{}"
        );
        newState.userVoted =
          userVotesFromStorage[data.currentRound] !== undefined;
        newState.userVotes = userVotesFromStorage;

        return newState;
      });
    } catch (error) {
      console.error("API fetch error:", error);
    }
  }, []);

  const submitVote = async (choice: string) => {
    try {
      if (gameState.userVoted) return;

      const response = await fetch(`${EXPRESS_BACKEND_URL}/api/vote`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ choice }),
      });

      if (!response.ok) throw new Error("Vote submission failed");

      const newUserVotes = {
        ...gameState.userVotes,
        [gameState.currentRound]: choice,
      };
      sessionStorage.setItem("userVotes", JSON.stringify(newUserVotes));

      setGameState((prev) => ({
        ...prev,
        userVotes: newUserVotes,
        userVoted: true,
      }));

      await fetchGameState();
    } catch (error) {
      console.error("Error submitting vote:", error);
    }
  };

  useEffect(() => {
    fetchGameState();
    const interval = setInterval(fetchGameState, 1000);
    return () => clearInterval(interval);
  }, [fetchGameState]);

  return { gameState, fetchGameState, submitVote };
}
