import Image from "next/image";
import React, { useEffect, useRef } from "react";

export type GameResult = "PLAYER" | "BANKER" | "TIE" | "DRAGON" | "PANDA";

interface TopGridBoardProps {
  results: GameResult[];
  stats: {
    label: string;
    count: number;
    iconImage: string;
    alt: string;
    color?: string;
  }[];
}

const PLAYER_MAIN_OUTCOME = "PLAYER";
const BANKER_MAIN_OUTCOME = "BANKER";

export const TopGridBoard: React.FC<TopGridBoardProps> = ({
  results,
  stats,
}) => {
  const rows = 6;
  const maxColumns = 36;
  const scrollContainerRef = useRef<HTMLDivElement>(null);

  const CELL_DIM_PX = 52; // Each cell is 52x52 px
  const MIN_VISIBLE_COLUMNS = 7; // Minimum columns to be visible without scroll
  const minContainerWidthPx = MIN_VISIBLE_COLUMNS * CELL_DIM_PX;

  const GRID_CONTENT_HEIGHT_PX = rows * CELL_DIM_PX; // 6 rows * 52px/row = 312px
  const CONTAINER_BORDER_THICKNESS_PX = 2; // Assuming border-2 means 2px border
  const CONTAINER_TOTAL_VERTICAL_BORDER_PX = 2 * CONTAINER_BORDER_THICKNESS_PX; // 4px for top & bottom
  const FINAL_CONTAINER_HEIGHT_PX =
    GRID_CONTENT_HEIGHT_PX + CONTAINER_TOTAL_VERTICAL_BORDER_PX; // 312px + 4px = 316px

  // Calculate actual columns needed based on results
  const actualColumns = Math.min(Math.ceil(results.length / rows), maxColumns);
  const columnsToShow = Math.max(actualColumns, 7); // At least 7 columns for minimum view

  // Auto-scroll to show most recent results
  const lastResultsLengthRef = useRef(0);

  useEffect(() => {
    if (
      scrollContainerRef.current &&
      results.length > 0 &&
      results.length !== lastResultsLengthRef.current
    ) {
      const container = scrollContainerRef.current;
      container.scrollLeft = container.scrollWidth - container.clientWidth;
      lastResultsLengthRef.current = results.length;
    }
  }, [results]);

  const gridCells = Array(rows * columnsToShow).fill(null);

  // Fill the grid with the most recent results that fit
  const maxCells = rows * columnsToShow;
  const resultsToShow =
    results.length > maxCells ? results.slice(-maxCells) : results;

  resultsToShow.forEach((result, index) => {
    const col = Math.floor(index / rows);
    const row = index % rows;
    const flatIndex = row * columnsToShow + col;

    let content = null;
    switch (result) {
      case PLAYER_MAIN_OUTCOME:
        content = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/player.png"
              alt="Player"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case BANKER_MAIN_OUTCOME:
        content = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/banker.png"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "TIE":
        content = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/TIEGif2.gif"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "DRAGON":
        content = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/dragonGIF.gif"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      case "PANDA":
        content = (
          <div className="relative w-[9vw] h-[9vw] sm:w-11 sm:h-11">
            <Image
              src="/images/pandaGIF.gif"
              alt="Banker"
              fill
              className="object-contain"
              unoptimized
            />
          </div>
        );
        break;
      default:
        break;
    }

    if (flatIndex < gridCells.length) {
      gridCells[flatIndex] = content;
    }
  });

  const gridRows = [];
  for (let row = 0; row < rows; row++) {
    const rowCells = [];
    for (let col = 0; col < columnsToShow; col++) {
      const index = row * columnsToShow + col;
      rowCells.push(
        <div
          key={`top-grid-${row}-${col}`}
          className=" w-[8.52vw] h-[8.52vw] sm:w-[56px] sm:h-[56px] xl:w-[55px] xl:h-[55px] flex-shrink-0 border border-[#292722] flex items-center justify-center relative"
          style={{ backgroundColor: "#f5e9d0" }}
        >
          {gridCells[index]}
        </div>
      );
    }
    gridRows.push(
      <div key={`top-grid-row-${row}`} className="flex">
        {rowCells}
      </div>
    );
  }

  return (
    <div className="flex mb-1 w-full">
      {/* Grid Section */}
      <div
        ref={scrollContainerRef}
        className={`  h-[${FINAL_CONTAINER_HEIGHT_PX}px] overflow-x-auto scroll-container border-2 border-amber-600`}
      >
        <div className="flex flex-col sm:w-fit">{gridRows}</div>
      </div>

      {/* Stats Section */}
      <div
        className={`sm:h-[${FINAL_CONTAINER_HEIGHT_PX}px] flex-1 flex-col sm:items-stretch`}
      >
        {/* Top Stats */}
        <div
          className="flex-1 bg-cover   bg-center   py-3 px-2 sm:py-4 xl:py-5 sm:px-3 flex-col h-full gap-2 flex flex-column justify-evenly"
          style={{ backgroundImage: "url('/images/bg-5.jpg');" }}
        >
          {stats.slice(0, 2).map((item, index) => (
            <div
              key={index}
              className="flex items-center justify-between  text-white px-3 md:px-3 py-0.5 sm:py-1.5 rounded text-sm font-bold shadow-sm"
            >
              <div className="flex items-center gap-2 ">
                {item.iconImage ? (
                  <div className="size-7 customScreenWidth:size-9 sm:size-11  sm:w-10 md:w-11 lg:w-12 relative">
                    <Image
                      src={item.iconImage}
                      alt={item.alt}
                      layout="fill"
                      objectFit="contain"
                      unoptimized
                    />
                  </div>
                ) : (
                  <div className={`w-4 h-4 rounded-full   ${item.color}`} />
                )}
                <span
                  className="text-[4vw]  sm:text-2xl xl:text-3xl font-impact mr-2 text-shadow-sm "
                  style={{
                    color: item.color ?? "#fff",
                  }}
                >
                  {item.label}
                </span>
              </div>
              <span className=" text-lg sm:text-2xl xl:text-3xl font-impact text-shadow-sm ">
                {item.count}
              </span>
            </div>
          ))}
          {stats.slice(2).map((item, index) => (
            <div
              key={index}
              className="flex items-center justify-between text-white px-3 md:px-3 py-0.5 sm:py-1.5 rounded text-sm font-bold shadow-sm"
            >
              <div className="flex items-center gap-2">
                {item.iconImage ? (
                  <div className="size-7 customScreenWidth:size-9 sm:size-11  sm:w-10 md:w-11 lg:w-12 relative">
                    <Image
                      src={item.iconImage}
                      alt={item.alt}
                      layout="fill"
                      objectFit="contain"
                      unoptimized
                    />
                  </div>
                ) : (
                  <div className={`w-4 h-4 rounded-full ${item.color}`} />
                )}
                <span
                  className="text-[4vw] sm:text-2xl xl:text-3xl font-impact mr-2 text-shadow-sm "
                  style={{
                    color: item.color ?? "#fff",
                  }}
                >
                  {item.label}
                </span>
              </div>
              <span className="text-lg sm:text-2xl xl:text-3xl font-impact text-shadow-sm ">
                {item.count}
              </span>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};
